/*****************************************************************************
 *
 * Copyright (C) 2002 Polytechnic University of Valencia.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Carlos Calafate, <calafate@disca.upv.es>
 *
 *****************************************************************************/


#include "PICA/timer.h"

struct prioq *pq;
HANDLE timerHandle = NULL;
DWORD WINAPI SigThr(LPVOID lpParameter);
static HANDLE timerThr = NULL;
HANDLE timerMutex = NULL;
HANDLE threadEvent = NULL;

UINT64 base_time = 0;

DWORD WINAPI TimerThrFunc(LPVOID lpParameter);


UINT64 PICAgetCurrTime(void)
{
	SYSTEMTIME st;
	FILETIME ft;
	UINT64 retval, aux2;

//creates SYSTEMTIME for now and January 1, 1970

	GetLocalTime(&st);

//converts to file time

	if (!SystemTimeToFileTime(&st,&ft)) {
	    P_ERROR("PICAgetCurrTime: Error converting system time to file time", 80101);	
		return 0;
	}


//	Converts to 64 bits

	aux2 = ft.dwHighDateTime;
	aux2 <<= 32;
	aux2 += ft.dwLowDateTime;	
	
	retval = (aux2-base_time) / 10000;

	return retval;
}


int PICAtimer(int action, UINT64 * time, void * function, void * data) {

	struct prioqent * curr_pq;


	//printf("In PICAtimer\n");

	switch (action) {
	case T_STARTUP: if (timerMutex == NULL) {
						P_ERROR("PICAtimer: PICA library not initialized", 80201);
						return 0;
					}
					if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {
						P_ERROR("PICAtimer: Error locking mutex", 80202);
						return 0;				  
					}

					if (pq != NULL) {
						P_ERROR("PICAtimer: Timer queue is not NULL", 80202);
						return 0;
					}

					if ((pq = (struct prioq*)malloc(sizeof(struct prioq))) == NULL) {
						P_ERROR("PICAtimer: Unable to allocate memory", 80203);
						return 0;
					}
					pq->pqe = NULL;
					if (!ReleaseMutex(timerMutex)) {
						P_ERROR("PICAtimer: Error unlocking mutex", 80204);	  
						return 0;
					}
					break;

  case T_SET:	if (pq == NULL) {
	                 P_ERROR("PICAtimer: timer queue was not initialized", 80205);
		             return 0;
				}
				if (time == NULL) {
					P_ERROR("PICAtimer: time points to NULL", 80206);
					return 0;
				}
				if (function == NULL) {
					P_ERROR("PICAtimer: function points to NULL", 80207);
					return 0;
				}
				if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {
						P_ERROR("PICAtimer: Error locking mutex", 80208);
						return 0;				  
				}
				if (!pq_insert(*time,function,data)) {
					P_ERROR("PICAtimer: Error inserting in queue", 80209);
					return 0;
				}	
				if (!ReleaseMutex(timerMutex)) {
						P_ERROR("PICAtimer: Error unlocking mutex", 80210);	  
						return 0;
				}
				break;

  case T_STOP:  if (pq == NULL) {
					P_ERROR("PICAtimer: timer queue was not initialized", 80211);
					return 0;
				}
				if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {
						P_ERROR("PICAtimer: Error locking mutex", 80212);
						return 0;				  
				}
	            if ((curr_pq = pq_getfromqueue(time, function)) == NULL) {
					P_ERROR("PICAtimer: Unable to find a matching element in the queue", 80213);
					return 0;
				}
				pq_deleteent(curr_pq);
				if (!ReleaseMutex(timerMutex)) {
						P_ERROR("PICAtimer: Error unlocking mutex", 80214);	  
						return 0;
				}
				break;

  case T_KILL:	if (pq == NULL) {
                 P_ERROR("PICAtimer: timer queue was not initialized", 80215);
                 return 0;
				}

				if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {
					P_ERROR("PICAtimer: Unable to lock mutex", 80216);	
					return 0;				  							 
				}
				if (!TerminateThread(timerThr,0)) {
					P_ERROR("PICAtimer: Unable to cancel timer thread", 80217);	
					return 0;
				}
				if (!CloseHandle(timerThr)) {
					P_ERROR("PICAtimer: Unable to close handle", 80214);
					return 0; 
				}
				timerThr = NULL;				
				if (!CancelWaitableTimer(timerHandle)) {
					P_ERROR("PICAtimer: Unable to cancel timer", 80215);
					return 0;
				}
				if (!CloseHandle(timerHandle)) {
					P_ERROR("PICAtimer: Unable to close handle", 80216);
					return 0; 
				}
				timerHandle = NULL;
				pq_cleanup();
				if (!ReleaseMutex(timerMutex)) {
					P_ERROR("PICAtimer: Unable to release mutex", 80217);					
					return 0;
				}
				
				break;

	default: {
				P_ERROR("PICAtimer: Unknown user action", 80218);
				return 0;    
			 }

  }

  return 1;
}





struct prioqent * pq_getfromqueue(UINT64 * time, void * function) {
  struct prioqent *pqe;
  
  /* Get first entry */
  pqe = pq->pqe;
  
  
  if (time != NULL) {
    if (function != NULL) {
      while (pqe != NULL && (pqe->tv != *time || pqe->callback != function))
	pqe=pqe->pqe;
    } else {
      while (pqe != NULL && pqe->tv != *time)
	pqe=pqe->pqe;
    }                
  } else {
    if (function != NULL) {
      while (pqe != NULL &&  pqe->callback != function)
	pqe=pqe->pqe;
    } else {
      return NULL;
    }  
  }
  
  return pqe;
}

void pq_cleanup(void) {
  struct prioqent * pqe, * pqe2;
  
  /* Get first entry */
  pqe = pq->pqe;
  
  while (pqe != NULL) {
    pqe2 = pqe->pqe;
    free(pqe);
    pqe=pqe2;
  }
  
  free(pq);
  pq = NULL;
}



/* 
 *   pq_updatetimer
 *
 *   Description: 
 *     Update the timer to the value of the event which
 *     is most recent. 
 *
 *   Arguments: None
 *
 *   Return: None
*/
void
pq_updatetimer()
{
  struct itimerval ovalue;
  struct itimerval value;
  UINT64 currtime;
  UINT64 tv;
  DWORD thrID;
  ULARGE_INTEGER liDueTime;
  HANDLE localTimerHandle = NULL;

  if (timerThr == NULL) {
	timerThr = CreateThread(NULL,0,TimerThrFunc,NULL,0,&thrID);
	if (timerThr == NULL) {
		return;	
	}
  }


  /* Init timer values to zero */
  (ovalue.it_interval).tv_sec = 0;
  (ovalue.it_interval).tv_usec = 0;
  
  (value.it_interval).tv_sec = 0;
  (value.it_interval).tv_usec = 0;


  if (pq->pqe == NULL)
    {
      /* No event to set timer for */
      (value.it_value).tv_sec = 0;
      (value.it_value).tv_usec = 0;
    }
  else 
    {
      /*  Get the first time value */
      tv = (pq->pqe)->tv;
      
      currtime = PICAgetCurrTime();
      
      if (tv <= currtime)
	{
	  /* If the event has allready happend, set the timeout to 
	     1 microsecond :-) */
	  (value.it_value).tv_sec = 0;
	  (value.it_value).tv_usec = 1;
	}
      else 
	{
	  /* Set the timer to the actual seconds / microseconds from now */
	  (value.it_value).tv_sec = (long) (tv - currtime) / 1000;
	  (value.it_value).tv_usec = (long) ((tv - currtime) % 1000) * 1000;
	}
      
    }
  
	liDueTime.QuadPart = (value.it_value).tv_sec * 10000000 + (value.it_value).tv_usec * 10;

	while (timerHandle == NULL) {
		//Sleep(1);
		WaitForSingleObject(threadEvent,INFINITE);
	}


	localTimerHandle = OpenWaitableTimer(TIMER_ALL_ACCESS,TRUE,"mytimer1");
	
	if (localTimerHandle == NULL) return;//printf("Error en OpenWaitableTimer\n");

	if (SetWaitableTimer(localTimerHandle,&liDueTime.QuadPart,liDueTime.QuadPart/10000,NULL,NULL,TRUE) == 0) {
		//printf("Error en SetWaitbleTimer\n");
		return; //quitar
	}


    return;
  
}



/* 
 *   pq_insert
 *
 *   Description: 
 *     Inserts an entry into the prioqueue.
 *
 *   Arguments: 
 *     UINT64 msec  - The time in milliseconds (msec from 1 jan 1970) of 
 *                       which the event shall occur.
 *     void *data  - Pointer to the data that shall be stored.
 *     unsigned int id  - A number to identyfy the stored data (like an 
 *                       ipnumber or something)
 *     unsigned char flags  - A flag that decribes what is stored in the 
 *                            data pointer 
 *
 *   Return: 
 *     int   - On error returns -1 else 0
*/
int pq_insert(UINT64 msec,void (*func)(void *),void * data) {
  struct prioqent *next_pqe;
  struct prioqent *prev_pqe;
  struct prioqent *pqe;
  
  /* get memory */
  if ((pqe = (struct prioqent*)malloc(sizeof(struct prioqent))) == NULL)
    /* Failed to allocate memory */
    return 0;

  /* copy data */
  pqe->tv = msec;
  pqe->callback = func;
  pqe->data = data;
  
  /* find a place for the new entry */
  
  /* is the queue empty or we are the first element */
  if (pq->pqe == NULL || (pq->pqe)->tv > pqe->tv)
    {
      pqe->pqe = pq->pqe;
      pq->pqe = pqe;
    }
  else 
    {
      /* we are further down in the list */
      next_pqe = (pq->pqe)->pqe;
      prev_pqe = pq->pqe;
      
      /* Go down while we are not at the end and all elements are due 
	 sooner than us */
      while (next_pqe != NULL && 
	     (prev_pqe->tv < pqe->tv && (next_pqe->tv) < pqe->tv) )
	{
	  prev_pqe = next_pqe;
	  next_pqe = next_pqe->pqe;
	}
      /* Reached the end of the list */
      if (next_pqe == NULL)
	{
	  prev_pqe->pqe = pqe;
	  pqe->pqe = NULL;
	}
      /* Found a place to insert it into */
      else 
	{
	  pqe->pqe = next_pqe;
	  prev_pqe->pqe = pqe;
	}
      
    }
  /* Update the timer to reflect the new situation */
  pq_updatetimer();
  
  return 1;
}



/* 
 *   pq_getfirst
 *
 *   Description: 
 *     Returns a pointer to the first entry in the queue.
 *
 *   Arguments: None
 *
 *   Return: 
 *     struct prioqent* - A ponter to the first entry in the queue.
 *                        Can be NULL if the queue is empty.
*/
struct prioqent * pq_getfirst() {
  return pq->pqe;
}


/* 
 *   pq_deleteent
 *
 *   Description: 
 *     Deletes the entry from the queue that is pointed to 
 *     by the argument. The entry is freed.
 *
 *   Arguments: 
 *     struct prioqent *pqed - A pointer to the entry that shall be deleted.
 *
 *   Return: None
*/
void pq_deleteent(struct prioqent *pqed) {
  struct prioqent *pqe;
  
  /* Is first element */
  if (pq->pqe == pqed)
    {
      pq->pqe = pqed->pqe;
      free(pqed);
    }
  else 
    {
      /* Not the first element in the list */
      pqe = pq->pqe;
      while (pqe->pqe != NULL && pqe->pqe != pqed)
	pqe = pqe->pqe;
      
      if (pqe != NULL)
	{
	  pqe->pqe = pqed->pqe;
	  free(pqed);
	}
    }
  
  pq_updatetimer();
}



/* 
 *   pq_unqueuefirstent
 *
 *   Description: 
 *     Remove but no delete the first entry in the queue. (not freed)
 *
 *   Arguments: None
 *
 *   Return: None
*/
void pq_unqueuefirstent() {
  struct prioqent *pqe;

  if (pq->pqe != NULL)
    {
      pqe = pq->pqe;
      pq->pqe = pqe->pqe;
    }

  pq_updatetimer();
}

/* 
 *   pq_deletefirstent
 *
 *   Description: 
 *     Deletes the first entry in the queue. 
 *
 *   Arguments: None
 *
 *   Return: None
*/
void pq_deletefirstent() {
  struct prioqent *pqe;

  if (pq->pqe != NULL)
    {
      pqe = pq->pqe;
      pq->pqe = pqe->pqe;
      free(pqe);
    }
  
  pq_updatetimer();
}

/* 
 *   pq_getfirstdue
 *
 *   Description: 
 *     Returns the first entry in the queue that has a time 
 *     that is lower than the argument ie. the first element
 *     if the argument is greater than its tv value.
 *
 *   Arguments: 
 *     UINT64 tv - the time that the elment to be returned shall be 
 *                    lower than. This is not quite true due to the real
 *                    time timer vs. the timeslice for the process the value
 *                    shound be in an interval +/- TIME_DIV to the time
 *                    tv given.
 *
 *   Return: 
 *     struct prioqent* - Pointer to the first entry if the time tv was
 *                        right else NULL is returned.
*/
struct prioqent * pq_getfirstdue(UINT64 tv) {
  struct prioqent *pqe;
  
  if ((pqe = pq_getfirst(pq)) != NULL)
    {
      /* If pqe's time is in teh interval */
      if ((pqe->tv) < tv + TIME_DIV || (pqe->tv) < tv - TIME_DIV)
	return pqe;
      
    }
  
  return NULL;
}



DWORD WINAPI TimerThrFunc(LPVOID lpParameter) {
    
	struct prioqent * pqe;
	UINT64 currtime;
  
	
	
	if ((timerHandle = CreateWaitableTimer(NULL,FALSE,"mytimer1")) == NULL) return 1;

	SetEvent(OpenEvent(EVENT_ALL_ACCESS,FALSE,"TimerThrEv1")); //notice main thread that the thread has started

	while (1) {
		WaitForSingleObject(timerHandle,INFINITE);	

	  /* Get the first due entry in the queue */
	  currtime = PICAgetCurrTime();

	  if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {					
		return 0;				  
	  }	            
	   
	  pqe = pq_getfirstdue(currtime);
  
	  if (!ReleaseMutex(timerMutex)) {			
			return 0;
	  }
	  /* While there is still events that has timed out */
	  while (pqe != NULL) {

		  pqe->callback(pqe->data);

	 	  if (WaitForSingleObject(timerMutex,INFINITE) != WAIT_OBJECT_0) {					
			return 0;				  
		  }	            

		  /* Dequeue the entry so that it will not happened again */
		  pq_unqueuefirstent();
      
		  /* Get new time and check for more timedout entrys */
		  currtime = PICAgetCurrTime();
      
		  pqe = pq_getfirstdue(currtime);
	  	  if (!ReleaseMutex(timerMutex)) {			
			return 0;
		  }

	  }
	}
		
}
