/*****************************************************************************
 *
 * Copyright (C) 2002 Polytechnic University of Valencia.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Carlos Calafate, <calafate@disca.upv.es>
 *
 *****************************************************************************/

#include "timer.h"



pthread_mutex_t timermutex = PTHREAD_MUTEX_INITIALIZER; 
sem_t * timersem;
pthread_t  timerthr; 

struct prioq *pq;

//timer thread definition
void TimerThrFunc(void * lpParameter);


UINT64 PICAgetCurrTime(void)
{
  struct timezone tz;
  struct timeval tv;
  
  if (gettimeofday(&tv, &tz) < 0) {
    /* Couldn't get time of day */
    P_ERROR("PICAgetCurrTime: Error getting system time", 80101);
    return 0;
  }
  
  return ((u_int64_t)tv.tv_sec) * 1000 + ((u_int64_t)tv.tv_usec) / 1000;
}


int PICAtimer(int action, UINT64 * time, void (*function)(void *), void * data) {
  
  struct prioqent * curr_pq;
  
  switch (action) {
  case T_STARTUP:  if (pthread_mutex_lock (&timermutex) != 0) {
                      P_ERROR("PICAtimer: Error locking mutex", 80201);
                      return 0;  
                   }
                   if (pq != NULL) {
		     P_ERROR("PICAtimer: Timer queue is not NULL", 80202);
		     return 0;
		   }
		   if ((pq = (struct prioq*)malloc(sizeof(struct prioq))) == NULL) {
		     P_ERROR("PICAtimer: Error allocating memory", 80203);
		     return 0;
		   }
		   pq->pqe = NULL;
		   if (sem_init(timersem,0,0) != 0) {
		     P_ERROR("PICAtimer: Error initializing timer semaphore", 80204);
		     return 0;
		   }
		   
		   if (signal(SIGALRM,pq_signal) == SIG_ERR) {
		     P_ERROR("PICAtimer: Error setting callback function", 80205);
		     return 0;		   
		   }
		   
		   if (pthread_create(&timerthr, NULL, TimerThrFunc, NULL) != 0) {
		     P_ERROR("PICAtimer: Error starting timer thread", 80206);
		     return 0; 
		   }
		   
		   if (pthread_mutex_unlock (&timermutex) != 0) {
		     P_ERROR("PICAtimer: Error unlocking mutex", 80207);
		     return 0;   
		   }
		   break;

  case T_SET:  if (pq == NULL) {
                 P_ERROR("PICAtimer: timer queue was not initialized", 80208);
                 return 0;
               }
               if (time == NULL) {
                 P_ERROR("PICAtimer: time points to NULL", 80209);
                 return 0;
               }
	       if (function == NULL) {
                 P_ERROR("PICAtimer: function points to NULL", 80210);
                 return 0;
	       }
	       if (pthread_mutex_lock (&timermutex) != 0) {
                      P_ERROR("PICAtimer: Error locking mutex", 80211);
                      return 0;  
	       }
               if (!pq_insert(*time,function,data)) {
		 P_ERROR("PICAtimer: Error inserting in queue", 80212);
		 return 0;
	       }
	       if (pthread_mutex_unlock (&timermutex) != 0) {
		 P_ERROR("PICAtimer: Error unlocking mutex", 80213);
		 return 0;   
	       }
	       break;

  case T_STOP: if (pq == NULL) {
                  P_ERROR("PICAtimer: timer queue was not initialized", 80214);
		  return 0;
               }   
               if (pthread_mutex_lock (&timermutex) != 0) {
		 P_ERROR("PICAtimer: Error locking mutex", 80215);
		 return 0;  
	       }         
               if ((curr_pq = pq_getfromqueue(time, function)) == NULL) {
		 P_ERROR("PICAtimer: Unable to find a matching element in the queue", 80216);
		 return 0;
	       }
	       pq_deleteent(curr_pq);			       
	       if (pthread_mutex_unlock (&timermutex) != 0) {
		 P_ERROR("PICAtimer: Error unlocking mutex", 80217);
		 return 0;   
	       }
	       break;

  case T_KILL: if (pq == NULL) {
                 P_ERROR("PICAtimer: timer queue was not initialized", 80218);
                 return 0;
               }
	       if (pthread_mutex_lock (&timermutex) != 0) {
		 P_ERROR("PICAtimer: Unable to lock mutex", 80214);
		 return 0;
	       }
	       if (pthread_cancel(timerthr) != 0) {
		 P_ERROR("PICAtimer: Unable to cancel timer thread", 80215);
		 return 0;  
	       }
	       timerthr = NULL;
	       if (sem_destroy(timersem) != 0) {
		 P_ERROR("PICAtimer: Unable to destroy semaphore", 80216);
		 return 0; 
		}
	       pq_cleanup();
	       if (pthread_mutex_unlock (&timermutex) != 0) {
		 P_ERROR("PICAtimer: Unable to unlock mutex", 80217);
		 return 0;
	       }
	       break;
	       
  default: {
	      P_ERROR("PICAtimer: Unknown user action", 80218);
	      return 0;    
	    }
  }

  return 1;
}


 
 
struct prioqent * pq_getfromqueue(UINT64 * time, void * function) {
  struct prioqent *pqe;
  
  /* Get first entry */
  pqe = pq->pqe;
  
  
  if (time != NULL) {
    if (function != NULL) {
      while (pqe != NULL && (pqe->tv != *time || pqe->callback != function))
	pqe=pqe->pqe;
    } else {
      while (pqe != NULL && pqe->tv != *time)
	pqe=pqe->pqe;
    }                
  } else {
    if (function != NULL) {
      while (pqe != NULL &&  pqe->callback != function)
	pqe=pqe->pqe;
    } else {
      return NULL;
    }  
  }
  
  return pqe;
}

void pq_cleanup(void) {
  struct prioqent * pqe, * pqe2;
  
  /* Get first entry */
  pqe = pq->pqe;
  
  while (pqe != NULL) {
    pqe2 = pqe->pqe;
    free(pqe);
    pqe=pqe2;
  }
  
  free(pq);
  pq = NULL;
}





/* 
 *   pq_updatetimer
 *
 *   Description: 
 *     Update the timer to the value of the event which
 *     is most recent. 
 *
 *   Arguments: None
 *
 *   Return: None
*/
void
pq_updatetimer()
{
  struct itimerval ovalue;
  struct itimerval value;
  u_int64_t currtime;
  u_int64_t tv;

  /* Init timer values to zero */
  (ovalue.it_interval).tv_sec = 0;
  (ovalue.it_interval).tv_usec = 0;
  
  (value.it_interval).tv_sec = 0;
  (value.it_interval).tv_usec = 0;


  if (pq->pqe == NULL)
    {
      /* No event to set timer for */
      (value.it_value).tv_sec = 0;
      (value.it_value).tv_usec = 0;
    }
  else 
    {
      /*  Get the first time value */
      tv = (pq->pqe)->tv;
      
      currtime = PICAgetCurrTime();
      
      if (tv <= currtime)
	{
	  /* If the event has allready happend, set the timeout to 
	     1 microsecond :-) */
	  (value.it_value).tv_sec = 0;
	  (value.it_value).tv_usec = 1;
	}
      else 
	{
	  /* Set the timer to the actual seconds / microseconds from now */
	  (value.it_value).tv_sec =  (tv - currtime) / 1000;
	  (value.it_value).tv_usec = ((tv - currtime) % 1000) * 1000;
	}
      
    }
  
  /* Set the timer (in real time) */
  if (setitimer(ITIMER_REAL,&value,&ovalue) < 0)
    /* Error */
    return;
  
}


/* 
 *   pq_signal
 *
 *   Description: 
 *     Is run when the timer signals that an event has happend.
 *     It gets the entrys from the queue that is due and writes
 *     their addresses to the pipe.
 *
 *   Arguments: 
 *     int signal - The number of the signal that caused it to be called.
 *
 *   Return: None
*/
void
pq_signal(int signal)
{
  //increase semaphore      
  sem_post(timersem);  
}

/* 
 *   pq_insert
 *
 *   Description: 
 *     Inserts an entry into the prioqueue.
 *
 *   Arguments: 
 *     u_int64_t msec  - The time in milliseconds (msec from 1 jan 1970) of 
 *                       which the event shall occur.
 *     void *data  - Pointer to the data that shall be stored.
 *     u_int32_t id  - A number to identyfy the stored data (like an 
 *                       ipnumber or something)
 *     unsigned char flags  - A flag that decribes what is stored in the 
 *                            data pointer 
 *
 *   Return: 
 *     int   - On error returns -1 else 0
*/

int pq_insert(UINT64 msec,void (*func)(void *),void * data) {
  struct prioqent *next_pqe;
  struct prioqent *prev_pqe;
  struct prioqent *pqe;
  
  /* get memory */
  if ((pqe = (struct prioqent*)malloc(sizeof(struct prioqent))) == NULL)
    /* Failed to allocate memory */
    return 0;

  /* copy data */
  pqe->tv = msec;
  pqe->callback = func;
  pqe->data = data;

  /* find a place for the new entry */
  
  /* is the queue empty or we are the first element */
  if (pq->pqe == NULL || (pq->pqe)->tv > pqe->tv)
    {
      pqe->pqe = pq->pqe;
      pq->pqe = pqe;
    }
  else 
    {
      /* we are further down in the list */
      next_pqe = (pq->pqe)->pqe;
      prev_pqe = pq->pqe;
      
      /* Go down while we are not at the end and all elements are due 
	 sooner than us */
      while (next_pqe != NULL && 
	     (prev_pqe->tv < pqe->tv && (next_pqe->tv) < pqe->tv) )
	{
	  prev_pqe = next_pqe;
	  next_pqe = next_pqe->pqe;
	}
      /* Reached the end of the list */
      if (next_pqe == NULL)
	{
	  prev_pqe->pqe = pqe;
	  pqe->pqe = NULL;
	}
      /* Found a place to insert it into */
      else 
	{
	  pqe->pqe = next_pqe;
	  prev_pqe->pqe = pqe;
	}
      
    }
  /* Update the timer to reflect the new situation */
  pq_updatetimer();
  
  return 1;
}




/* 
 *   pq_getfirst
 *
 *   Description: 
 *     Returns a pointer to the first entry in the queue.
 *
 *   Arguments: None
 *
 *   Return: 
 *     struct prioqent* - A ponter to the first entry in the queue.
 *                        Can be NULL if the queue is empty.
*/
struct prioqent * pq_getfirst()
{
  return pq->pqe;
}



/* 
 *   pq_deleteent
 *
 *   Description: 
 *     Deletes the entry from the queue that is pointed to 
 *     by the argument. The entry is freed.
 *
 *   Arguments: 
 *     struct prioqent *pqed - A pointer to the entry that shall be deleted.
 *
 *   Return: None
*/
void
pq_deleteent(struct prioqent *pqed)
{
  struct prioqent *pqe;
  
  /* Is first element */
  if (pq->pqe == pqed)
    {
      pq->pqe = pqed->pqe;
      free(pqed);
    }
  else 
    {
      /* Not the first element in the list */
      pqe = pq->pqe;
      while (pqe->pqe != NULL && pqe->pqe != pqed)
	pqe = pqe->pqe;
      
      if (pqe != NULL)
	{
	  pqe->pqe = pqed->pqe;
	  free(pqed);
	}
    }
  
  pq_updatetimer();
}




/* 
 *   pq_unqueuefirstent
 *
 *   Description: 
 *     Remove but no delete the first entry in the queue. (not freed)
 *
 *   Arguments: None
 *
 *   Return: None
*/
void
pq_unqueuefirstent()
{
  struct prioqent *pqe;

  if (pq->pqe != NULL)
    {
      pqe = pq->pqe;
      pq->pqe = pqe->pqe;
    }

  pq_updatetimer();
}

/* 
 *   pq_deletefirstent
 *
 *   Description: 
 *     Deletes the first entry in the queue. 
 *
 *   Arguments: None
 *
 *   Return: None
*/
void
pq_deletefirstent()
{
  struct prioqent *pqe;

  if (pq->pqe != NULL)
    {
      pqe = pq->pqe;
      pq->pqe = pqe->pqe;
      free(pqe);
    }
  
  pq_updatetimer();
}

/* 
 *   pq_getfirstdue
 *
 *   Description: 
 *     Returns the first entry in the queue that has a time 
 *     that is lower than the argument ie. the first element
 *     if the argument is greater than its tv value.
 *
 *   Arguments: 
 *     u_int64_t tv - the time that the elment to be returned shall be 
 *                    lower than. This is not quite true due to the real
 *                    time timer vs. the timeslice for the process the value
 *                    shound be in an interval +/- TIME_DIV to the time
 *                    tv given.
 *
 *   Return: 
 *     struct prioqent* - Pointer to the first entry if the time tv was
 *                        right else NULL is returned.
*/

struct prioqent * pq_getfirstdue(u_int64_t tv) {
  struct prioqent *pqe;

  if ((pqe = pq_getfirst(pq)) != NULL) {
      /* If pqe's time is in the interval */
	if ((pqe->tv) < tv + TIME_DIV) {	   
	    return pqe;
	}
    }
  
  return NULL;
}


/* 
 *   pq_print
 *
 *   Description: 
 *     Prints the prio queue.
 *
 *   Arguments: None
 * 
 *  Return: None

void
pq_print()
{
  struct prioqent *pqe;

  pqe = pq->pqe;
  while (pqe != NULL)
    {
      printf("sec/msec: %lu/%lu id:%lu\n", (unsigned long)(pqe->tv) / 1000,
	     (unsigned long)(pqe->tv) % 1000, (unsigned long)pqe->id);
      pqe = pqe->pqe;
    }
  
}
*/
void TimerThrFunc(void * lpParameter) {  
  
  struct prioqent * pqe;
  UINT64 currtime;
  
  
  while (1) {

    /* Wait for timer event */
    sem_wait(timersem); 

    
    /* Get the first due entry in the queue */
    currtime = PICAgetCurrTime();

    if (pthread_mutex_lock (&timermutex) != 0) {
      return;  
    }

    //perform security check!

    if ((pqe = pq_getfirstdue(currtime)) == NULL) { //should not happend!
	if ((pqe = pq_getfirst(pq)) != NULL) { //there are entries	   
	    pq_updatetimer(); //update the timer	    
	}
	if (pthread_mutex_unlock (&timermutex) != 0)  return;   
	
	continue; //restart loop
    }
  
    
    if (pthread_mutex_unlock (&timermutex) != 0) {
      return;   
    }
    
    /* While there is still events that has timed out */
    while (pqe != NULL)
      {
	pqe->callback(pqe->data);
	if (pthread_mutex_lock (&timermutex) != 0) {
	  return;  
	}
	/* Remove the entry so that it will not happened again */
	pq_deletefirstent();
	
	/* Get new time and check for more timedout entrys */
	currtime = PICAgetCurrTime();
	
	pqe = pq_getfirstdue(currtime);
	if (pthread_mutex_unlock (&timermutex) != 0) {
	  return;   
	}
      }
    
  }
  
}


