/*****************************************************************************
 *
 * Copyright (C) 2002 Polytechnic University of Valencia.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Carlos Calafate, <calafate@disca.upv.es>
 *
 *****************************************************************************/

#include "thr.h"


typedef struct {
    int		inuse;				/* 1 if in use, else 0 */
    pthread_t	id;				/* Thread ID */
    } Victim_t;

sem_t sem;
pthread_mutex_t mut = PTHREAD_MUTEX_INITIALIZER;
pthread_once_t once = PTHREAD_ONCE_INIT;
Victim_t *array = NULL;
int bottom = 0;
int init_routine_retval = 1;

/*
 * Handle SIGUSR1 in the target thread, to suspend it until receiving
 * SIGUSR2 (resume). Note that this is run with both SIGUSR1 and
 * SIGUSR2 blocked. Having SIGUSR2 blocked prevents a resume before we
 * can finish the suspend protocol.
 */
void suspend_signal_handler (int sig) {
    sigset_t signal_set;

    /*
     * Block all signals except SIGUSR2 while suspended.
     */
    sigfillset (&signal_set);
    sigdelset (&signal_set, SIGUSR2);
    sem_post (&sem);
    sigsuspend (&signal_set);

    /*
     * Once here, the resume signal handler has run to completion.
     */
    return;
}

/*
 * Handle SIGUSR2 in the target thread, to resume it. Note that
 * the signal handler does nothing. It exists only because we need
 * to cause sigsuspend() to return.
 */
void
resume_signal_handler (int sig)
{
    return;
}

/*
 * Dynamically initialize the "suspend package" when first used
 * (called by pthread_once).
 */
void
suspend_init_routine (void)
{
   int status;
    struct sigaction sigusr1, sigusr2;

    /*
     * Initialize a semaphore, to be used by the signal handler to
     * confirm suspension. We only need one, because suspend & resume
     * are fully serialized by a mutex.
     */
    if ((status = sem_init (&sem, 0, 1)) == -1) init_routine_retval = 0;
    
    //        errno_abort ("Initializing semaphore");

    /*
     * Allocate the suspended threads array. This array is used to guarantee
     * idempotency.
     */
   
    bottom = 10;
   
    array = (Victim_t*) calloc (bottom, sizeof (Victim_t));

    /*
     * Install the signal handlers for suspend/resume.
     *
     * We add SIGUSR2 to the sa_mask field for the SIGUSR1 handler. That
     * avoids a race if one thread suspends the target while another resumes
     * that same target. (The SIGUSR2 signal cannot be delivered before the
     * target thread calls sigsuspend.)
     */
    sigusr1.sa_flags = 0;
    sigusr1.sa_handler = suspend_signal_handler;
    sigemptyset (&sigusr1.sa_mask);
    sigaddset (&sigusr1.sa_mask, SIGUSR2);

    sigusr2.sa_flags = 0;
    sigusr2.sa_handler = resume_signal_handler;
    sigemptyset (&sigusr2.sa_mask);

    if ((status = sigaction (SIGUSR1, &sigusr1, NULL)) == -1) init_routine_retval = 0;
      //  errno_abort ("Installing suspend handler");
    
    if ((status = sigaction (SIGUSR2, &sigusr2, NULL)) == -1) init_routine_retval = 0;;
    //errno_abort ("Installing resume handler");

    return;
}



int PICAstartThread(THRID * thr, void * func, void * arg) {
  pthread_attr_t detach;

  if (thr == NULL) {
    P_ERROR("PICAstartThread: thread pointer is NULL", 60101);
    return 0;
  }

  if (func == NULL) {
    P_ERROR("PICAstartThread: function pointer is NULL", 60102);
    return 0;
  }
  
  if (pthread_attr_init (&detach) != 0) {
    P_ERROR("PICAstartThread: Error initializing attributes", 60103);
    return 0;
  }
  
  if (pthread_attr_setdetachstate (&detach,PTHREAD_CREATE_DETACHED ) != 0) {// or PTHREAD_CREATE_JOINABLE
    P_ERROR("PICAstartThread: Error setting detach state", 60104);
    return 0;
  }

  if (pthread_create(thr, &detach, func, arg) != 0) {
    P_ERROR("PICAstartThread: Error creating thread", 60105);
    return 0;
  }
  
  return 1;
}


int PICAsuspendThread(THRID target_thread) {

  //   int status;
    int i;

    /*
     * The first call to thd_suspend will initialize the
     * package.
     */

    if(pthread_once(&once, suspend_init_routine) != 0) {
      P_ERROR("PICAsuspendThread: Error calling pthread_once", 60201);
      return 0;
    }
    if (init_routine_retval == 0) {
      P_ERROR("PICAsuspendThread: Error executing initialization routine", 60202);
      return 0;
    }

    /* 
     * Serialize access to suspend, makes life easier
     */
    if (pthread_mutex_lock(&mut) != 0) {
      P_ERROR("PICAsuspendThread: Error locking mutex", 60203);
      return 0;
    }


    /*
     * Threads that are suspended are added to the target_array; a request to
     * suspend a thread already listed in the array is ignored.
     */
    for (i = 0; i < bottom; i++) {
        if (array[i].inuse && pthread_equal (array[i].id, target_thread)) {
	    if (pthread_mutex_unlock(&mut) != 0) {
	      P_ERROR("PICAsuspendThread: Error unlocking mutex", 60204);
	    } else P_ERROR("PICAsuspendThread: Thread already in suspend state", 60205);
            return 0;
	}
    }

    /*
     * Ok, we really need to suspend this thread. So, lets find the location
     * in the array that we'll use. If we run off the end, realloc the array
     * for more space. We allocate 2 new array slots; we'll use one, and leave 
     * the other for the next time.
     */
    i = 0;
    while (i < bottom && array[i].inuse)
        i++;

    if (i >= bottom) {
	bottom += 2;
        array = (Victim_t*) realloc (
            array, (bottom * sizeof (Victim_t)));
        if (array == NULL) {
	  if (pthread_mutex_unlock(&mut) != 0) {
	    P_ERROR("PICAsuspendThread: Error unlocking mutex", 60206);
	  } else P_ERROR("PICAsuspendThread: Error realocating memory", 60207);
	  return 0;
        }
        array[bottom-1].inuse = 0;	/* Clear new last entry */
    }
    
    /*
     * Initialize the target's data. We initialize a semaphore to synchronize
     * with the signal handler. After sending the signal, we wait until the
     * signal handler posts the semaphore. Then we can destroy the semaphore,
     * because we won't need it again.
     */
    array[i].id = target_thread;
    array[i].inuse = 1;

    if (pthread_kill (target_thread, SIGUSR1) != 0) {
        if (pthread_mutex_unlock(&mut) != 0) {
	  P_ERROR("PICAsuspendThread: Error unlocking mutex", 60208);
	} else P_ERROR("PICAsuspendThread: Error sending SIGUSR1 to thread", 60209);
	return 0;
    }

    /*
     * Wait for the victim to acknowledge suspension.
     */
    while (sem_wait (&sem) != 0) {
      if (errno != EINTR) {
	if (pthread_mutex_unlock(&mut) != 0) {
	  P_ERROR("PICAsuspendThread: Error unlocking mutex", 60210);
	} else P_ERROR("PICAsuspendThread: Error waiting for semaphore", 60211);
	return 0;
      }
    }
    
    if (pthread_mutex_unlock (&mut) != 0) {
      P_ERROR("PICAsuspendThread: Error unlocking mutex", 60212);
      return 0;
    }
    
    return 1;
}


int PICAresumeThread(THRID target_thread) {

    int i = 0;

    /*
     * The first call to thd_suspend will initialize the package.
     */
    if (pthread_once (&once, suspend_init_routine) != 0)  {
      P_ERROR("PICAresumeThread: Error executing initialization routine", 60301);
      return 0;
    }


    /* 
     * Serialize access to suspend, makes life easier.
     */
    if (pthread_mutex_lock (&mut) != 0) {
      P_ERROR("PICAresumeThread: Error locking mutex", 60302);
      return 0;
    }


    /*
     * Make sure the thread is in the suspend array. If not, it hasn't been
     * suspended (or it has already been resumed) and we can just carry on.
     */
    while (i < bottom && array[i].inuse && pthread_equal (array[i].id, target_thread))
        i++;

    if (i >= bottom) {
      if (pthread_mutex_unlock (&mut) != 0) {
	P_ERROR("PICAresumeThread: Error unlocking mutex", 60303);
	
      } else P_ERROR("PICAresumeThread: Can not resume thread", 60304);
      return 0;
    }

    /*
     * Signal the thread to continue, and remove the thread from
     * the suspended array.
     */
    if (pthread_kill (target_thread, SIGUSR2) != 0) {
      if (pthread_mutex_unlock (&mut) != 0) {
	P_ERROR("PICAresumeThread: Error unlocking mutex", 60304);
      } else P_ERROR("PICAresumeThread: Error sending signal SIGUSR2 to thread", 60305);
      return 0;
    }

    array[i].inuse = 0;			/* Clear array element */

    if (pthread_mutex_unlock (&mut) != 0) {
      P_ERROR("PICAresumeThread: Error unlocking mutex", 60306);
      return 0;
    }

  return 1;
}


int PICAkillThread(THRID thr) {
  
  if (pthread_cancel(thr) != 0) {
    P_ERROR("PICAkillThread: Error calling pthread_cancel", 60401);
    return 0;  
  }

  return 1;
}


int PICAaddDesc(PICAdescList ** dl, int type, void * desc) {
 
        PICAdescList * runner, * newdesc;


	if (dl == NULL) {	 
	  P_ERROR("PICAaddDesc: Descriptor list pointer is NULL", 60501);
	  return 0;
	}
 
        if ((newdesc = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
	  P_ERROR("PICAaddDesc: Error allocating memory", 60502);
	  return 0;
	}

        newdesc->type = type;
        newdesc->desc = desc;
        newdesc->next = NULL; 
        
	if (*dl == NULL) {
	  *dl = newdesc;
        } else {
	  runner = *dl;	  
	  while (runner->next != NULL) runner = runner->next;	  
	  runner ->next = newdesc;
        }
 
        return 1;
}


int PICAselect(int time, PICAdescList * dl, PICAselResult * res) {
 PICAdescList * runner;
  struct timeval to, * sel_time;
  fd_set rread;
  int end, *curr_sock;
  int sr, max = 0, * sock;

 
  if (dl == NULL) {
    P_ERROR("PICAselect: Descriptor list pointer is NULL", 60601);
    return 0;
  }
  
  if (res == NULL) {
    P_ERROR("PICAselect: select result pointer is NULL", 60602);
    return 0;
  }


  runner = dl;
 
  FD_ZERO(&rread);
 
  while (runner != NULL) {
    sock = (int *)  runner->desc;
    FD_SET(*sock,&rread); // indicate that we want to
    if (*sock > max) max = *sock;
    runner = runner->next;
  }
  max++;

  memset((char *)&to,0,sizeof(to)); // clear the timeval struct
  
  if (time == PICA_WAIT_FOREVER) sel_time = NULL;
  else {
    to.tv_sec= (long) time/1000;                // timeout select after 5 secs
    to.tv_usec = (long) (time - to.tv_sec*1000)*1000;
    sel_time = &to;
  }
  
  if ((sr=select(max, &rread, (fd_set *)0, (fd_set *)0, sel_time)) < 0) {
    switch (errno) {
    case EBADF: P_ERROR("PICAselect: Invalid descriptor found", 60603);
      break;
    case EINTR: P_ERROR("PICAselect: Non-blocking signal receive", 60604);
      break;
    case EINVAL: P_ERROR("PICAselect: PICA internal error for select - negative descriptor value", 60605);
      break;
    case ENOMEM: P_ERROR("PICAselect: select was unable to allocate enough memory", 60606);
      break;
    default: P_ERROR("PICAselect: Error in select function", 60607);
    }
    return 0;
  } else if (sr == 0) {
    res->type = PICA_TIMEOUT_TYPE;
    res->desc = NULL;
    
  } else  {
    
    runner = dl;
    end = 0;
    
    while (runner != NULL && !end) {
      curr_sock = (int *) runner->desc;
      if (FD_ISSET(*curr_sock,&rread))
        {
          res->type = runner->type;
          res->desc = runner->desc;
          end = 1;
	}
      runner = runner->next;
    }
  }
  return 1;
}


int PICAcreateMutex(PICAmutex * mut) {
  
  if (mut == NULL) {
    P_ERROR("PICAcreateMutex: mutex pointer is NULL", 60701);
    return 0;
  }
  
  pthread_mutex_init(mut, NULL);
  
  return 1;
}

int PICAmutexAction(int action, PICAmutex * mut) {
  
  if (mut == NULL) {
    P_ERROR("PICAmutexAction: mutex pointer is NULL", 60801);
    return 0;
  }
  
  switch (action) {
    
  case MUTEX_ACQUIRE: if (pthread_mutex_lock(mut) != 0) {
    P_ERROR("PICAmutexAction: Error locking mutex", 60802);
    return 0;
  }
  break;
  case MUTEX_RELEASE:  if (pthread_mutex_unlock(mut) != 0) {
    P_ERROR("PICAmutexAction: Error unlocking mutex", 60803);
    return 0;
  } 
  break;
  case MUTEX_ACQ_NO_BLOCK: if (pthread_mutex_trylock(mut) != 0) {
    P_ERROR("PICAmutexAction: Error trying to lock mutex", 60804);
    return 0;
  }
  break;
  default: {
    P_ERROR("PICAmutexAction: Unknown user action", 60805);
    return 0;
    }
    
  }
  
  return 1;
}

int PICAdestroyMutex(PICAmutex * mut) {
  
  if (mut == NULL)  {
    P_ERROR("PICAdestroyMutex: mutex pointer is NULL", 60901);
    return 0;
  }

  if (pthread_mutex_destroy(mut) != 0) {
    P_ERROR("PICAdestoyMutex: Error trying to destroy mutex", 60902);
    return 0;
  }
  
  return 1;
}


int PICAcreateSemaphore(PICAsemaphore * p_sem, int initial_count, int max_count) {    

  if (p_sem == NULL)  {
    P_ERROR("PICAcreateSemaphore: PICAsemaphore pointer is NULL", 61001);
    return 0;
  }
  
  if (initial_count < 0) {
    P_ERROR("PICAcreateSemaphore: invalid initial count value", 61002);
    return 0;
  }

  if (max_count < 1 || max_count < initial_count) {
    P_ERROR("PICAcreateSemaphore: invalid max count value", 61003);
    return 0;
  }


  if (sem_init(&(p_sem->sem),0,initial_count) != 0) {
    P_ERROR("PICAcreateSemaphore: Error initializing semaphore", 61004);
    return 0;
  }
  if (pthread_mutex_init(&(p_sem->sem_mutex), NULL) != 0) {
    P_ERROR("PICAcreateSemaphore: Error initializing mutex", 61005);
    return 0;
  }
  p_sem->max_count = max_count;
  return 1;
}


int PICAsemaphoreAction(int action, PICAsemaphore * p_sem, int count) {
  int i,j, svalue, top;

  
  if (p_sem == NULL)  {
    P_ERROR("PICAsemaphoreAction: PICAsemaphore pointer is NULL", 61101);
    return 0;
  }
  
  if (count < 1) {
    P_ERROR("PICAsemaphoreAction: invalid value for count", 61102);
    return 0;
  }

  
  switch (action) {
    
  case SEMAPHORE_ACQUIRE: 
    for (i=0;i<count; i++) {                         
      if (sem_wait(&(p_sem->sem)) != 0) {
	for (j=0;j<i;j++) sem_post(&(p_sem->sem));
	P_ERROR("PICAsemaphoreAction: Error waiting for semaphore", 61103);
	return 0;
      }
    }
    break;
    
  case SEMAPHORE_RELEASE: 

    if (pthread_mutex_lock(&(p_sem->sem_mutex)) != 0) {
      P_ERROR("PICAsemaphoreAction: Error trying to lock mutex", 61104);
      return 0;
    }

    sem_getvalue(&(p_sem->sem), &svalue);

    if (p_sem->max_count >= (svalue + count) ) top = count;
    else top = p_sem->max_count - svalue;


    for (i=0;i<top;i++) {
	if (sem_post(&(p_sem->sem)) != 0) {
	  P_ERROR("PICAsemaphoreAction: Error incrementing semaphore", 61105);
	  return 0;
	}
    }
    
    if (pthread_mutex_unlock(&(p_sem->sem_mutex)) != 0) {
      P_ERROR("PICAsemaphoreAction: Error trying to unlock mutex", 61106);
      return 0;
    }
    break;

  case SEMAPHORE_ACQ_NO_BLOCK: 
    for (i=0;i<count; i++) {                         
      if (sem_trywait(&(p_sem->sem)) != 0) {
	for (j=0;j<i;j++) sem_post(&(p_sem->sem));
	P_ERROR("PICAsemaphoreAction: Error waiting for semaphore", 61107);
	return 0;
      }
    }
    break;
  default: P_ERROR("PICAsemaphoreAction: Unknown user action", 61108);
    return 0;
    
  }
  
  return 1;
}

int PICAdestroySemaphore(PICAsemaphore * p_sem) {
  

  if (p_sem == NULL)  {
    P_ERROR("PICAsemaphoreAction: PICAsemaphore pointer is NULL", 61201);
    return 0;
  }
  
  if (sem_destroy(&(p_sem->sem)) != 0) {
    P_ERROR("PICAdestroySemaphore: Error destroying semaphore", 61202);
    return 0;
  }
  
  return 1;
}

