/*****************************************************************************
 *
 * Copyright (C) 2002 Polytechnic University of Valencia.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Carlos Calafate, <calafate@disca.upv.es>
 *
 *****************************************************************************/

#include "mem.h"


int PICAinitBuffer(PICAbuffer ** ibuf, int num_queues) {

  int i;
  PICAbuffer * buf;

  if (ibuf == NULL)  {
    P_ERROR("PICAinitBuffer: ibuf is NULL", 70101);
    return 0;
  }


  if ((*ibuf = (PICAbuffer *) malloc(sizeof(PICAbuffer))) == NULL) {
    P_ERROR("PICAinitBuffer: Unable to allocate memory", 70102);
    return 0;
  }

  if (num_queues < 1) {
    P_ERROR("PICAinitBuffer: Number of queues must be greater than 0", 70103);
    return 0;
  }


  buf = *ibuf;

  buf->tot_queues = num_queues;
  if ((buf->buf_mut = (pthread_mutex_t *) malloc(num_queues*sizeof(pthread_mutex_t))) == NULL) {
    P_ERROR("PICAinitBuffer: Unable to allocate memory", 70104);
    return 0;
  }
  
  for (i=0;i<num_queues;i++) {
    pthread_mutex_init(&(buf->buf_mut[i]), NULL);
  }

  if ((buf->packet_queues = (struct _PICApacket **) malloc(num_queues*sizeof(struct _PICApacket *))) == NULL) {
    P_ERROR("PICAinitBuffer: Unable to allocate memory", 70105);
    return 0;
  }
  memset(buf->packet_queues, 0, num_queues*sizeof(struct _PICApacket *));
  //bzero(buf->packet_queues, num_queues*sizeof(struct _PICApacket *));

  return 1;
}

int PICAaddToBuffer(PICAbuffer * buf, int queue_id, void * data, int data_size) {
  PICApacket * curr_pkt, * runner;


 if (buf == NULL) {
   P_ERROR("PICAaddToBuffer: Buffer points to NULL", 70201);
   return 0;
 }

 if (data_size <= 0) {
   P_ERROR("PICAaddToBuffer: Data size must be greater than 0", 70202);
   return 0;
 }

 if (data == NULL) {
   P_ERROR("PICAaddToBuffer: Data points to NULL", 70203);
   return 0;
 }

 if (queue_id < 0 || queue_id >= buf->tot_queues) {
   P_ERROR("PICAaddToBuffer: Invalid queue number", 70204);
   return 0;
 }

 if ((curr_pkt = (PICApacket *) malloc(sizeof(PICApacket))) == NULL) {
   P_ERROR("PICAaddToBuffer: Error allocating memory", 70205);
   return 0;
 }
 
  curr_pkt->packet_size = data_size;
  curr_pkt->data = data;
  curr_pkt->next = NULL;
  
  if (pthread_mutex_lock(&(buf->buf_mut[queue_id])) != 0) {
    P_ERROR("PICAaddToBuffer: Unable to lock mutex", 70206);
    return 0;
  }
  //start protected sequence

  runner = buf->packet_queues[queue_id];

  // if (runner != NULL) printf("ADD: First Data: %s\n",(char *) runner->data);

  if (runner == NULL) {
    buf->packet_queues[queue_id] = curr_pkt;
    //    printf("IS FIRST\n");
  }
  else {
    while (runner->next != NULL) runner = runner->next;  
    runner->next = curr_pkt;
  }

  runner = buf->packet_queues[queue_id];
  while (runner != NULL) {
    //   printf("ADD: Total queue Data: %s\n",(char *) runner->data);
    runner = runner->next;
  }

  //end protected sequence
  if (pthread_mutex_unlock(&(buf->buf_mut[queue_id])) != 0) {
    P_ERROR("PICAaddToBuffer: Unable to unlock mutex", 70207);
    return 0;
  }

  return 1;
}

int PICAgetFromBuffer(PICAbuffer * buf, int queue_id, int num_packets, PICApacket ** packets, int * avail_packets) {

 PICApacket * runner;


 if (buf == NULL) {
   P_ERROR("PICAgetFromBuffer: Buffer points to NULL", 70301);
   return 0;
 }

 if (packets == NULL) {
   P_ERROR("PICAgetFromBuffer: packets points to NULL", 70302);
   return 0;
 }

 if (num_packets <= 0) {
   P_ERROR("PICAgetFromBuffer: number of packets must be greater than 0", 70303);
   return 0;
 }

 if (queue_id < 0 || queue_id >= buf->tot_queues) {
   P_ERROR("PICAgetFromBuffer: Invalid queue identifier", 70304);
   return 0;
 }

 // printf("GET: start num_packets = %d and avail_packets %d\n",num_packets, *avail_packets);


 //printf("GET: Starting lock\n");


 *packets = NULL;
 *avail_packets = 0;


  if (pthread_mutex_lock(&(buf->buf_mut[queue_id])) != 0) {
    P_ERROR("PICAgetFromBuffer: Unable to lock mutex", 70305);
    return 0;
  }
  //start protected sequence
 
  runner = buf->packet_queues[queue_id];

 *packets = runner;

  if (runner != NULL) {
    num_packets--;

    (*avail_packets) ++;

    while (runner->next != NULL && num_packets != 0) {
      //      printf("GET: in while with num_packets = %d\n",num_packets);
      runner = runner->next;  
      num_packets--;
      (*avail_packets)++;
    }
    buf->packet_queues[queue_id] = runner->next;
    runner->next = NULL;
  }
  
  //end protected sequence
  if (pthread_mutex_unlock(&(buf->buf_mut[queue_id])) != 0) {
    P_ERROR("PICAgetFromBuffer: Unable to unlock mutex", 70306);
    return 0;
  }

  return 1;
}

int PICAkillBuffer(PICAbuffer * buf) {
  int i, num_queues;
  PICApacket * curr_pos, * next_pos;
  pthread_mutex_t * mut_copy;


  
  if (buf == NULL) {
    P_ERROR("PICAkillBuffer: Buffer points to NULL", 70401);
    return 0;
  }
  

  for (i=0;i<buf->tot_queues; i++) {
    if (pthread_mutex_trylock(&(buf->buf_mut[i])) != 0) {
      P_ERROR("PICAkillBuffer: Error while trying to lock mutex", 70402);
      return 0;
    }
  }



  mut_copy = buf->buf_mut;
  num_queues = buf->tot_queues;



  for (i=0;i<buf->tot_queues; i++) {
    curr_pos  = buf->packet_queues[i];
    buf->packet_queues[i] = NULL;
    

    
    while (curr_pos != NULL) {
      next_pos = curr_pos->next;
      free(curr_pos->data);  //posible error se memoria no alocada
      free(curr_pos);
      curr_pos = next_pos;
    }
  }
  


  free(buf->packet_queues);
  
  buf->buf_mut = NULL;
  

  free(buf);
  
  buf = NULL;
  

  for (i=0;i<num_queues; i++) {
    if (pthread_mutex_unlock(&(mut_copy[i])) != 0) {
      P_ERROR("PICAkillBuffer: Error while trying to unlock mutex", 70403);
      return 0;
    }
  }

  
 free(mut_copy);
 
 return 1;
 
}


int PICAmakePipe(PICApipe * in, PICApipe * out) {
  int desc[2];

  if (in == NULL) {
      P_ERROR("PICAmakePipe: pipe pointer for input is NULL", 70501);
      return 0;
    }

  if (out == NULL) {
      P_ERROR("PICAmakePipe: pipe pointer for output is NULL", 70502);
      return 0;
  }

  if (pipe(desc) == -1) {
    P_ERROR("PICAmakePipe: Error creating pipe", 70503);
    return 0;
  }
  *in = desc[0];
  *out = desc[1];

  return 1;
}
 
int PICAsendToPipe(PICApipe out, void * data, int size, int * written) {
  
  if (data == NULL)  {
    P_ERROR("PICAsendToPipe: data points to NULL", 70601);
    return 0;
  }

  if (written == NULL)  {
    P_ERROR("PICAsendToPipe: written points to NULL", 70602);
    return 0;
  }
  

  if ((*written = write(out, data, size)) == -1) {
    P_ERROR("PICAsendToPipe: Error writing to pipe", 70603);
    return 0;
  }

  return 1;
}
 
int PICAgetFromPipe(PICApipe in, void * buf, int bufsize, int * datasize) {

  if (buf == NULL)  {
    P_ERROR("PICAgetFromPipe: buffer points to NULL", 70701);
    return 0;
  }

  if (datasize == NULL)  {
    P_ERROR("PICAgetFromPipe: datasize points to NULL", 70702);
    return 0;
  }


  if ((*datasize = read(in, buf, bufsize)) == -1) {
    P_ERROR("PICAgetFromPipe: Error reading from pipe", 70703);
    return 0;
  }

  return 1;
}
 
int PICAclosePipe(PICApipe pipe) {
  
  if (close(pipe) == -1) {
    P_ERROR("PICAclosePipe: Error closing pipe", 70801);
    return 0;
  }
  return 1;
}
