/*****************************************************************************
 *
 * Copyright (C) 2002 Polytechnic University of Valencia.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors: Carlos Calafate, <calafate@disca.upv.es>
 *
 *****************************************************************************/


#include "thr.h"
#include "mem.h"

static HANDLE signalPipe;
static int timeout;
int sockselect_result = -1;


int PICAstartThread(THRID * thr, void * func, void * arg) {
	
	if (thr == NULL) {
		P_ERROR("PICAstartThread: thread pointer is NULL", 60101);
		return 0;
	}

	if (func == NULL) {
		P_ERROR("PICAstartThread: function pointer is NULL", 60102);
		return 0;
	}

	if ((*thr = CreateThread(NULL,0,(unsigned long (__cdecl *)(void *))func,arg,0,NULL)) == NULL) {
		P_ERROR("PICAstartThread: Error creating thread", 60103);
		return 0;
	}
	
	return 1;
}

int PICAsuspendThread(THRID thr) {

	if(SuspendThread(thr) == -1) {
		P_ERROR("PICAsuspendThread: Error suspending thread", 60201);
		return 0;
	}
	return 1;
}

int PICAresumeThread(THRID thr) {
	if(ResumeThread(thr) == -1) {
		P_ERROR("PICAresumeThread: Error resuming thread", 60301);
		return 0;
	}
	return 1;
}


int PICAkillThread(THRID thr) {	

	if (TerminateThread(thr,0) == 0) {
		P_ERROR("PICAkillThread: Error killing thread", 60401);
		return 0;
	}

	return 1;
}

int PICAaddDesc(PICAdescList ** dl, int type, int mode, void * desc) {
	
	PICAdescList * runner, * newdesc;	
	
	if (dl == NULL) {
		P_ERROR("PICAaddDesc: PICAdescList points to NULL", 60501);
		return 0;
	}

    if ((newdesc = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
		P_ERROR("PICAaddDesc: Error allocating memory", 60501);
		return 0;
	}	

	newdesc->type = type;

	newdesc->mode = mode;

	newdesc->desc = desc; 

	newdesc->next = NULL;
	
	if (*dl == NULL) {
		*dl = newdesc;
	} else {
		runner = *dl;

		while (runner->next != NULL) runner = runner->next;

		runner ->next = newdesc;
	}
	
	return 1;
}


DWORD WINAPI sock_select(LPVOID sdesc) {
	PICAdescList * socks, * runner;
	int sock_cnt, written, sock_num, end;
	char val[8];
	SOCKET * curr_sock;

	struct timeval to, * sel_time;
	fd_set rread;
	int socketopen=1;
	int sr;


	socks = (PICAdescList *) sdesc;

	runner = (PICAdescList *) sdesc;
	//runner = socks;
	sock_cnt = 0;
	FD_ZERO(&rread);	

	while (runner != NULL) {
		sock_cnt++;					
		curr_sock = (SOCKET *) (runner->desc);
		FD_SET(*curr_sock,&rread);
		runner = runner->next;
	}		
	

	if (timeout == PICA_WAIT_FOREVER) sel_time = NULL;
	else sel_time = &to;
    
    memset((char *)&to,0,sizeof(to)); // clear the timeval struct
    to.tv_sec= (long) timeout / 1000;	
	to.tv_usec = (long) (timeout - to.tv_sec*1000)/1000;
    
    // do the select.  
    // select returns > 0 if there is a read event on the socket 
    // e.g. data waiting to be read


    if ((sr=select(0, &rread, (fd_set *)0, (fd_set *)0, sel_time)) == SOCKET_ERROR) {
		
		sockselect_result = -1;
		SetEvent(signalPipe);
	}
	else if (sr == 0) {
		//Timeout in select. Do nothing
		
	} else {

		end = 0;
		sock_num = 0;
		runner = socks;
		while (runner != NULL && !end) {
			curr_sock = (SOCKET *) runner->desc;
			if (FD_ISSET(*curr_sock,&rread))
			  {
				sockselect_result = sock_num;
				SetEvent(signalPipe);
				end = 1;
			  }
			sock_num++;
			runner = runner->next;
		}

	}
	return 1;
}


int PICAselect(int time, PICAdescList * dl, PICAselResult * res) {

	PICAdescList * runner, * socks, *s_run, * handles, * h_run;
	HANDLE * handleArr, * aux;
	int sock_cnt=0, handle_cnt=0, curr_handle = 0, read, sockretval, i;
	SECURITY_ATTRIBUTES sa;
	HANDLE r_Pipe, thr;
	DWORD retval;
	char buf[8];
	fd_set rread;
	struct timeval to;
	SOCKET * curr_sock;
	int sr;

	
	FD_ZERO(&rread);

	if (dl == NULL) {
		P_ERROR("PICAselect: Descriptor list pointer is NULL", 60601);
		return 0;
	}
  
	if (res == NULL) {
		P_ERROR("PICAselect: select result pointer is NULL", 60602);
		return 0;
	}

	timeout = time;

	runner = dl;
	socks = NULL;
	handles = NULL;
	

	while (runner != NULL) {		
		if (runner->type == PICA_SOCKET_TYPE || runner->type == PICA_PIPE_TYPE) {
			sock_cnt++;
			if (socks == NULL) {
				if ((socks = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
					P_ERROR("PICAselect: error allocating memory", 60603);
					return 0;
				}
				s_run = socks;			
			}
			else {
				if ((s_run->next = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
					P_ERROR("PICAselect: error allocating memory", 60604);
					return 0;
				}
				s_run = s_run->next;				
			}
			
			memcpy(s_run,runner,sizeof(PICAdescList));
			//add to select group
			FD_SET(*((SOCKET *) runner->desc),&rread);
			s_run->next = NULL;

		} else if (runner->type == PICA_OTHER_TYPE) {
			handle_cnt++;
			if (handles == NULL) {
				if ((handles = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
					P_ERROR("PICAselect: error allocating memory", 60605);					
					return 0;
				}
				h_run = handles;			
			}
			else {
				if ((h_run->next = (PICAdescList *) malloc(sizeof(PICAdescList))) == NULL) {
					P_ERROR("PICAselect: error allocating memory", 60606);
					return 0;
				}
				h_run = h_run->next;				
			}
			
			memcpy(h_run,runner,sizeof(PICAdescList));
			h_run->next = NULL;
		} else {
			P_ERROR("PICAselect: Unknown type", 60607);						
			return 0;
		}

		runner = runner->next;
	}
	
	if ((handleArr = (PHANDLE) malloc((handle_cnt+1) * sizeof(HANDLE))) == NULL) {
		P_ERROR("PICAselect: error allocating memory", 60608);
		return 0;
	}

	if ((signalPipe = CreateEvent(NULL,FALSE,FALSE,NULL)) == NULL) {
			P_ERROR("PICAselect: error creating event", 60609);
			return 0;
	}

	handleArr[curr_handle++] = signalPipe;

	if (sock_cnt > 0) {			
		//check if a socket has already been activated
		to.tv_sec = 0;
		to.tv_usec = 0;

		if ((sr=select(0, &rread, (fd_set *)0, (fd_set *)0, &to)) == SOCKET_ERROR) {
			//handle error ---	
			P_ERROR("PICAselect: error in select. Perhaps an invalid socket exists", 60610);
			return 0;
		} else if (sr == 0) {
			//Timeout in select. Let the thread handle it			
			if ((thr = CreateThread(NULL,0,&sock_select, socks,0,NULL)) == NULL) {
				P_ERROR("PICAselect: error creating thread", 60611);
				return 0;				
			}
		} else {
			runner = socks;
			while (runner != NULL) {
				curr_sock = (SOCKET *) runner->desc;
				if (FD_ISSET(*curr_sock,&rread))
				  {					
					res->type = runner->type;
					res->desc = runner->desc;				
					return 1;
				  }			
				runner = runner->next;
			}	
		}			
	}

	runner = handles;
	
	while (runner != NULL) {
		aux = (PHANDLE) runner->desc;
		handleArr[curr_handle++] = *aux;
		runner = runner->next;		
	}

	if (time == PICA_WAIT_FOREVER) retval = WaitForMultipleObjects(handle_cnt+1,handleArr,FALSE,INFINITE);
	else retval = WaitForMultipleObjects(handle_cnt+1,handleArr,FALSE,time);

	if ((retval >= WAIT_OBJECT_0) && (retval <= WAIT_OBJECT_0+handle_cnt)) {
		
		if (retval == WAIT_OBJECT_0) {

			sockretval = sockselect_result;

			if (sockretval == -1) {
				P_ERROR("PICAselect: error in select function", 60612);	
				return 0;
			}
			
			runner = socks;
			
			for(i=0;i<sockretval;i++) runner = runner->next;
			res->type = runner->type;
			res->desc = runner->desc;

		} else {
			runner = handles;			
			for(i=0;i< retval-1-WAIT_OBJECT_0;i++) runner = runner->next;
			res->type = runner->type;
			res->desc = runner->desc;
		}

	} else if ((retval >= WAIT_ABANDONED_0) && (retval <= WAIT_ABANDONED_0 + handle_cnt)) {	
		P_ERROR("PICAselect: wait abandoned in WaitForMultipleObjects", 60613);
		return 0;
	} else if (retval == WAIT_TIMEOUT) {	
		res->type = PICA_TIMEOUT_TYPE;
	} else if (retval == WAIT_FAILED) {
		P_ERROR("PICAselect: wait failed in call to WaitForMultipleObjects", 60614);
		return 0;
	} else {
		P_ERROR("PICAselect: Unknown result from WaitForMultipleObjects", 60615);
		return 0;
	}

	return 1;
}


int PICAcreateMutex(PICAmutex * mut) {
	SECURITY_ATTRIBUTES sa;

	if (mut == NULL) {
		P_ERROR("PICAcreateMutex: mutex pointer is NULL", 60701);
		return 0;
	}


	sa.bInheritHandle = TRUE;
	sa.lpSecurityDescriptor = NULL;
	sa.nLength = sizeof(SECURITY_ATTRIBUTES);

	if ((*mut = CreateMutex(&sa,FALSE,NULL)) == NULL) {
		P_ERROR("PICAcreateMutex: error creating mutex", 60702);
		return 0;
	}

	return 1;
}

int PICAmutexAction(int action, PICAmutex * mut) {
	
	if (mut == NULL) {
		P_ERROR("PICAmutexAction: mutex pointer is NULL", 60801);
		return 0;
	}

	switch (action) {
		
	case MUTEX_ACQUIRE: if (WaitForSingleObject(*mut,INFINITE) != WAIT_OBJECT_0) {
							P_ERROR("PICAmutexAction: Error locking mutex", 60802);
							return 0;		
						}
						break;
	case MUTEX_RELEASE: if (!ReleaseMutex(*mut)) {
							P_ERROR("PICAmutexAction: Error unlocking mutex", 60803);
							return 0;
						}
						break;
	case MUTEX_ACQ_NO_BLOCK: if (WaitForSingleObject(*mut,0) != WAIT_OBJECT_0) {
								P_ERROR("PICAmutexAction: Error trying to lock mutex", 60805);
								return 0;		
							 }
							break;
	default: {
				P_ERROR("PICAmutexAction: unknown user action", 60806);
				return 0;
			 }

	}

	return 1;
}

int PICAdestroyMutex(PICAmutex * mut) {

	if (!CloseHandle(*mut)) {
		P_ERROR("PICAdestoyMutex: Error trying to destroy mutex", 60901);
		return 0;
	}

	return 1;
}


int PICAcreateSemaphore(PICAsemaphore * p_sem, int initial_count, int max_count) {

	if (p_sem == NULL)  {
		P_ERROR("PICAcreateSemaphore: PICAsemaphore pointer is NULL", 61001);
		return 0;
	}
  
	if (initial_count < 0) {
	    P_ERROR("PICAcreateSemaphore: invalid initial count value", 61002);
		return 0;
	}

	if (max_count < 1 || max_count < initial_count) {
	    P_ERROR("PICAcreateSemaphore: invalid max count value", 61003);
	    return 0;
	}


	if ((*p_sem = CreateSemaphore(NULL, initial_count,max_count, NULL)) == NULL) {
	    P_ERROR("PICAcreateSemaphore: Error creating semaphore", 61004);	
		return 0;
	}

	return 1;
}


int PICAsemaphoreAction(int action, PICAsemaphore * p_sem, int count) {
	int i;

	if (p_sem == NULL)  {
		P_ERROR("PICAsemaphoreAction: PICAsemaphore pointer is NULL", 61101);
		return 0;
	}
  
	if (count < 1) {
	    P_ERROR("PICAsemaphoreAction: invalid value for count", 61102);
	    return 0;
	}


	switch (action) {
		
	case SEMAPHORE_ACQUIRE: for (i=0;i<count; i++) {
								if (WaitForSingleObject(*p_sem,INFINITE) != WAIT_OBJECT_0) {
									ReleaseSemaphore(*p_sem,i,NULL);
									P_ERROR("PICAsemaphoreAction: Error waiting for semaphore", 61103);
									return 0;		
								}
							}
							break;
	case SEMAPHORE_RELEASE: if (!ReleaseSemaphore(*p_sem,count,NULL)) {
								P_ERROR("PICAsemaphoreAction: Error releasing semaphore", 61104);
								return 0;
							}
							break;
	case SEMAPHORE_ACQ_NO_BLOCK: for (i=0;i<count; i++) {
									if (WaitForSingleObject(*p_sem,0) != WAIT_OBJECT_0) {
										ReleaseSemaphore(*p_sem,i,NULL);
										P_ERROR("PICAsemaphoreAction: Error trying to get semaphore", 61105);
										return 0;		
									}
								}		
							break;
	default: {
				P_ERROR("PICAsemaphoreAction: unknown user action", 61106);
				return 0;
			 }

	}

	return 1;
}

int PICAdestroySemaphore(PICAsemaphore * p_sem) {

	if (p_sem == NULL)  {
	    P_ERROR("PICAsemaphoreAction: PICAsemaphore pointer is NULL", 61201);
		return 0;
	}

	if (!CloseHandle(*p_sem)) {
		P_ERROR("PICAdestroySemaphore: Error destroying semaphore", 61202);
		return 0;
	}
	
	return 1;
}